
import { createClient } from 'https://esm.sh/@supabase/supabase-js@2.43.3';

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
};

interface IncrementVoteParams {
  employee_id: string;
  user_email: string;
}

Deno.serve(async (req: Request) => {
  // Handle CORS preflight requests
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_SERVICE_ROLE_KEY') ?? ''
    );

    const { employee_id, user_email } = await req.json() as IncrementVoteParams;

    if (!employee_id || !user_email) {
      return new Response(
        JSON.stringify({ error: 'employee_id and user_email are required' }),
        { status: 400, headers: { 'Content-Type': 'application/json', ...corsHeaders } }
      );
    }

    // Check if the user has already voted
    const { data: existingVote, error: checkError } = await supabase
      .from('user_votes')
      .select('*')
      .eq('user_email', user_email)
      .maybeSingle();

    if (checkError) {
      return new Response(
        JSON.stringify({ error: 'Error checking existing votes', details: checkError }),
        { status: 500, headers: { 'Content-Type': 'application/json', ...corsHeaders } }
      );
    }

    if (existingVote) {
      return new Response(
        JSON.stringify({ error: 'User has already voted' }),
        { status: 400, headers: { 'Content-Type': 'application/json', ...corsHeaders } }
      );
    }

    // Perform transaction to increment vote and record user vote
    const { error: voteError } = await supabase.rpc('increment_vote', {
      employee_id
    });

    if (voteError) {
      return new Response(
        JSON.stringify({ error: 'Failed to increment vote', details: voteError }),
        { status: 500, headers: { 'Content-Type': 'application/json', ...corsHeaders } }
      );
    }

    // Record the user's vote
    const { error: recordError } = await supabase
      .from('user_votes')
      .insert([
        {
          user_email,
          employee_id,
          is_sent_confirmation: false
        }
      ]);

    if (recordError) {
      return new Response(
        JSON.stringify({ error: 'Failed to record user vote', details: recordError }),
        { status: 500, headers: { 'Content-Type': 'application/json', ...corsHeaders } }
      );
    }

    return new Response(
      JSON.stringify({ success: true, message: 'Vote recorded successfully' }),
      { status: 200, headers: { 'Content-Type': 'application/json', ...corsHeaders } }
    );
  } catch (error) {
    console.error('Error handling vote increment:', error);
    return new Response(
      JSON.stringify({ error: 'Internal server error', details: String(error) }),
      { status: 500, headers: { 'Content-Type': 'application/json', ...corsHeaders } }
    );
  }
});
