
import { create } from 'zustand';
import { persist } from 'zustand/middleware';
import { Employee, User, VotingData, AdminRole } from '../types';
import { supabase } from '@/integrations/supabase/client';

interface VotingStore {
  votingData: VotingData;
  currentUser: User | null;
  setVotingData: (data: VotingData) => void;
  openVoting: () => void;
  closeVoting: () => void;
  addEmployee: (employee: Employee) => Promise<void>;
  updateEmployee: (employee: Employee) => Promise<void>;
  deleteEmployee: (id: string) => Promise<void>;
  castVote: (employeeId: string) => Promise<void>;
  login: (email: string, password: string) => Promise<boolean>;
  authenticateVoter: (email: string) => Promise<boolean>;
  logout: () => void;
  checkUserVoteStatus: (email: string) => Promise<{ hasVoted: boolean, votedFor?: string }>;
  checkAdminStatus: (email: string) => Promise<AdminRole | null>;
}

// Mock closing date (30 days from now)
const closingDate = new Date();
closingDate.setDate(closingDate.getDate() + 30);

// Admin credentials - in a real app, this would be stored securely on the backend
const ADMIN_EMAIL = 'admin@company.com';
const ADMIN_PASSWORD = 'admin123';

export const useVotingStore = create<VotingStore>()(
  persist(
    (set, get) => ({
      votingData: {
        isOpen: true,
        closingDate: closingDate.toISOString(),
        employees: [],
      },
      currentUser: null,

      setVotingData: (data) => set({ votingData: data }),

      openVoting: () => set((state) => ({
        votingData: {
          ...state.votingData,
          isOpen: true,
        }
      })),

      closeVoting: () => set((state) => ({
        votingData: {
          ...state.votingData,
          isOpen: false,
        }
      })),

      addEmployee: async (employee) => {
        // Update the property name to match Supabase schema
        const supabaseEmployee = {
          ...employee,
          image_url: employee.image_url, // Ensure we use image_url not imageUrl
        };

        const { error } = await supabase
          .from('employees')
          .insert([{ ...supabaseEmployee, votes: 0 }]);

        if (error) throw error;
      },

      updateEmployee: async (employee) => {
        // Update the property name to match Supabase schema
        const supabaseEmployee = {
          ...employee,
          image_url: employee.image_url, // Ensure we use image_url not imageUrl
        };

        const { error } = await supabase
          .from('employees')
          .update(supabaseEmployee)
          .eq('id', employee.id);

        if (error) throw error;
      },

      deleteEmployee: async (id) => {
        const { error } = await supabase
          .from('employees')
          .delete()
          .eq('id', id);

        if (error) throw error;
      },

      castVote: async (employeeId) => {
        if (!get().currentUser || get().currentUser.hasVoted || !get().votingData.isOpen) {
          throw new Error("Cannot vote: Either not logged in, already voted, or voting is closed");
        }

        const currentUser = get().currentUser;
        if (!currentUser || !currentUser.email) {
          throw new Error("User email is required to vote");
        }

        // Call the Supabase Edge Function to increment vote
        const { error } = await supabase.functions.invoke('increment_vote', {
          body: { employee_id: employeeId, user_email: currentUser.email },
        });
        
        if (error) throw new Error(error.message);

        // Update local state to reflect user has voted
        set((state) => ({
          currentUser: {
            ...state.currentUser!,
            hasVoted: true,
            votedFor: employeeId
          }
        }));
      },

      login: async (email, password) => {
        // Check admin status
        const adminRole = await get().checkAdminStatus(email);
        
        // Set the user state with admin role information
        set({
          currentUser: {
            id: 'admin',
            email: email,
            hasVoted: false,
            isAdmin: !!adminRole,
            role: adminRole
          }
        });
        
        return true;
      },

      checkAdminStatus: async (email) => {
        // First, check if the email exists in the admins table
        const { data, error } = await supabase
          .from('admins')
          .select('role')
          .eq('email', email)
          .single();

        if (error && error.code !== 'PGRST116') {
          console.error("Error checking admin status:", error);
          return null;
        }

        return data ? data.role as AdminRole : null;
      },

      authenticateVoter: async (email) => {
        // In a real app, verify against backend database
        if (email && email.includes('@')) {
          // Check if this email has already voted
          const voteStatus = await get().checkUserVoteStatus(email);
          
          // Check admin status
          const adminRole = await get().checkAdminStatus(email);
          
          set({
            currentUser: {
              id: Math.random().toString(36).substr(2, 9),
              email,
              hasVoted: voteStatus.hasVoted,
              votedFor: voteStatus.votedFor,
              isAdmin: !!adminRole,
              role: adminRole
            }
          });
          return true;
        }
        return false;
      },

      checkUserVoteStatus: async (email) => {
        const { data, error } = await supabase
          .from('user_votes')
          .select('*')
          .eq('user_email', email)
          .single();

        if (error && error.code !== 'PGRST116') { // PGRST116 is "no rows returned" error
          console.error("Error checking vote status:", error);
          return { hasVoted: false };
        }

        return { 
          hasVoted: !!data, 
          votedFor: data ? data.employee_id : undefined 
        };
      },

      logout: () => set({ currentUser: null }),
    }),
    {
      name: 'voting-store',
    }
  )
);
