
import React, { useState } from 'react';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Search } from 'lucide-react';
import EmployeeCard from '@/components/EmployeeCard';
import { useVotingStore } from '../store/votingStore';
import { useEmployees } from '@/hooks/useEmployees';
import { useToast } from '@/components/ui/use-toast';

const VotePage = () => {
  const { currentUser, votingData } = useVotingStore();
  const { data: employees = [], isLoading, error, refetch } = useEmployees();
  const [searchQuery, setSearchQuery] = useState('');
  const { toast } = useToast();

  if (isLoading) {
    return <div className="text-center p-10">Loading employees...</div>;
  }

  if (error) {
    toast({
      title: "Error",
      description: "Failed to load employees. Please try again later.",
      variant: "destructive",
    });
    return <div className="text-center p-10">Error loading employees.</div>;
  }

  const filteredEmployees = employees.filter(employee =>
    employee.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    employee.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
    employee.description.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div>
      <div className="flex flex-col items-center mb-8">
        <h1 className="text-3xl font-bold mb-2">Cast Your Vote</h1>
        <p className="text-muted-foreground text-center max-w-2xl">
          {votingData.isOpen
            ? "Select an employee you believe deserves to be recognized as Employee of the Year 2024."
            : "Voting has closed. Thank you for your participation!"}
        </p>
        {currentUser?.hasVoted && (
          <div className="mt-4 p-3 bg-green-100 text-green-800 rounded-md text-center">
            You have already cast your vote. Thank you for participating!
          </div>
        )}
        {!currentUser && (
          <div className="mt-4 p-3 bg-amber-100 text-amber-800 rounded-md text-center">
            Please login to cast your vote.
          </div>
        )}
      </div>

      <div className="max-w-md mx-auto mb-8">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
          <Input
            type="text"
            placeholder="Search nominees..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-10"
          />
        </div>
      </div>

      {filteredEmployees.length > 0 ? (
        <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
          {filteredEmployees.map(employee => (
            <EmployeeCard 
              key={employee.id} 
              employee={employee} 
              onVoteSuccess={refetch}
            />
          ))}
        </div>
      ) : (
        <div className="text-center p-10">
          <p className="text-muted-foreground">No employees found matching your search.</p>
        </div>
      )}
    </div>
  );
};

export default VotePage;
