
import React from 'react';
import { useVotingStore } from '../store/votingStore';
import { Card, CardContent, CardHeader } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { Trophy, Award, Medal } from 'lucide-react';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { useEmployees } from '@/hooks/useEmployees';

const ResultsPage = () => {
  const { votingData } = useVotingStore();
  const { data: employees = [], isLoading } = useEmployees();
  const { isOpen } = votingData;
  
  if (isLoading) {
    return <div className="text-center p-10">Loading results...</div>;
  }
  
  // Sort employees by votes in descending order
  const sortedEmployees = [...employees].sort((a, b) => b.votes - a.votes);
  
  // Calculate total votes
  const totalVotes = employees.reduce((sum, employee) => sum + employee.votes, 0);
  
  // Get top 3 employees if available
  const topThree = sortedEmployees.slice(0, 3);

  const getInitials = (name: string) => {
    return name
      .split(' ')
      .map(word => word[0])
      .join('');
  };

  return (
    <div>
      <h1 className="text-3xl font-bold text-center mb-4">Employee of the Year 2024</h1>
      <p className="text-center text-muted-foreground mb-8">
        {isOpen 
          ? "Current standings - voting is still open!" 
          : "Final results are in! Congratulations to our winners!"}
      </p>
      
      {totalVotes > 0 ? (
        <>
          {/* Top 3 Winners Podium */}
          <div className="flex flex-col items-center mb-12">
            <h2 className="text-xl font-bold mb-6">Top Performers</h2>
            <div className="flex flex-col md:flex-row items-end justify-center gap-4 md:gap-8">
              {/* Second Place */}
              {topThree.length > 1 && (
                <div className="flex flex-col items-center">
                  <Avatar className="h-16 w-16 md:h-20 md:w-20 mb-2 ring-2 ring-gray-300">
                    <AvatarImage src={topThree[1].image_url} alt={topThree[1].name} />
                    <AvatarFallback>{getInitials(topThree[1].name)}</AvatarFallback>
                  </Avatar>
                  <div className="bg-gray-200 w-24 md:w-28 h-20 rounded-t-lg flex items-center justify-center">
                    <Medal className="h-8 w-8 text-gray-500" />
                  </div>
                  <div className="text-center mt-2">
                    <p className="font-bold">{topThree[1].name}</p>
                    <p className="text-sm text-muted-foreground">{topThree[1].title}</p>
                    <p className="text-sm font-semibold">{topThree[1].votes} votes</p>
                  </div>
                </div>
              )}
              
              {/* First Place */}
              {topThree.length > 0 && (
                <div className="flex flex-col items-center">
                  <div className="relative">
                    <Avatar className="h-20 w-20 md:h-24 md:w-24 mb-2 ring-2 ring-yellow-400">
                      <AvatarImage src={topThree[0].image_url} alt={topThree[0].name} />
                      <AvatarFallback>{getInitials(topThree[0].name)}</AvatarFallback>
                    </Avatar>
                    <Trophy className="absolute -top-4 -right-4 h-8 w-8 text-yellow-500" />
                  </div>
                  <div className="bg-yellow-100 w-28 md:w-32 h-28 rounded-t-lg flex items-center justify-center">
                    <Trophy className="h-10 w-10 text-yellow-500" />
                  </div>
                  <div className="text-center mt-2">
                    <p className="font-bold text-lg">{topThree[0].name}</p>
                    <p className="text-sm text-muted-foreground">{topThree[0].title}</p>
                    <p className="text-sm font-semibold">{topThree[0].votes} votes</p>
                  </div>
                </div>
              )}
              
              {/* Third Place */}
              {topThree.length > 2 && (
                <div className="flex flex-col items-center">
                  <Avatar className="h-16 w-16 md:h-20 md:w-20 mb-2 ring-2 ring-amber-700">
                    <AvatarImage src={topThree[2].image_url} alt={topThree[2].name} />
                    <AvatarFallback>{getInitials(topThree[2].name)}</AvatarFallback>
                  </Avatar>
                  <div className="bg-amber-100 w-24 md:w-28 h-16 rounded-t-lg flex items-center justify-center">
                    <Award className="h-8 w-8 text-amber-700" />
                  </div>
                  <div className="text-center mt-2">
                    <p className="font-bold">{topThree[2].name}</p>
                    <p className="text-sm text-muted-foreground">{topThree[2].title}</p>
                    <p className="text-sm font-semibold">{topThree[2].votes} votes</p>
                  </div>
                </div>
              )}
            </div>
          </div>
          
          {/* All Results */}
          <div>
            <h2 className="text-xl font-bold mb-4">Full Results</h2>
            <Card>
              <CardHeader className="pb-2">
                <div className="flex justify-between text-sm font-medium">
                  <span>Employee</span>
                  <span>Votes</span>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                {sortedEmployees.map((employee, index) => (
                  <div key={employee.id} className="space-y-2">
                    <div className="flex justify-between">
                      <div className="flex items-center gap-2">
                        <span className="text-sm font-medium text-muted-foreground w-4">
                          {index + 1}.
                        </span>
                        <Avatar className="h-8 w-8">
                          <AvatarImage src={employee.image_url} alt={employee.name} />
                          <AvatarFallback>{getInitials(employee.name)}</AvatarFallback>
                        </Avatar>
                        <span className="font-medium">{employee.name}</span>
                      </div>
                      <span className="font-medium">
                        {employee.votes} ({Math.round((employee.votes / totalVotes) * 100) || 0}%)
                      </span>
                    </div>
                    <Progress value={totalVotes ? (employee.votes / totalVotes) * 100 : 0} />
                  </div>
                ))}
              </CardContent>
            </Card>
          </div>
        </>
      ) : (
        <div className="text-center py-10">
          <p className="text-lg">No votes have been cast yet.</p>
          <p className="text-muted-foreground">Check back after the voting period begins!</p>
        </div>
      )}
    </div>
  );
};

export default ResultsPage;
