
import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { useToast } from '@/hooks/use-toast';
import { useVotingStore } from '../store/votingStore';
import { Employee } from '../types';
import AdminEmployeeForm from '@/components/AdminEmployeeForm';
import AdminAnalytics from '@/components/AdminAnalytics';
import AdminRoleForm from '@/components/AdminRoleForm';
import { useEmployees } from '@/hooks/useEmployees';
import AdminHeader from '@/components/admin/AdminHeader';
import EmployeeManagement from '@/components/admin/EmployeeManagement';
import VotingControls from '@/components/admin/VotingControls';

const AdminPage: React.FC = () => {
  const { votingData, currentUser, openVoting, closeVoting, addEmployee, updateEmployee, deleteEmployee } = useVotingStore();
  const { toast } = useToast();
  const navigate = useNavigate();
  const [isFormOpen, setIsFormOpen] = useState(false);
  const [isRoleFormOpen, setIsRoleFormOpen] = useState(false);
  const [currentEmployee, setCurrentEmployee] = useState<Employee | undefined>(undefined);
  const [activeTab, setActiveTab] = useState('employees');
  const { data: employees = [], refetch } = useEmployees();
  
  React.useEffect(() => {
    if (!currentUser?.isAdmin) {
      navigate('/');
      toast({
        title: 'Access Denied',
        description: 'You need admin privileges to access this page.',
        variant: 'destructive',
      });
    }
  }, [currentUser, navigate, toast]);

  const canManageEmployees = currentUser?.role === 'super_admin';
  const canManageVoting = currentUser?.role === 'super_admin' || currentUser?.role === 'voting_admin';
  const isViewerOnly = currentUser?.role === 'viewer';
  
  const handleAddEmployee = () => {
    if (!canManageEmployees) {
      toast({
        title: 'Permission Denied',
        description: 'You need super admin privileges to add employees.',
        variant: 'destructive',
      });
      return;
    }
    
    setCurrentEmployee(undefined);
    setIsFormOpen(true);
  };
  
  const handleEditEmployee = (employee: Employee) => {
    if (!canManageEmployees) {
      toast({
        title: 'Permission Denied',
        description: 'You need super admin privileges to edit employees.',
        variant: 'destructive',
      });
      return;
    }
    
    setCurrentEmployee(employee);
    setIsFormOpen(true);
  };
  
  const handleFormSubmit = async (employee: Partial<Employee>) => {
    if (currentEmployee) {
      await updateEmployee({ ...currentEmployee, ...employee } as Employee);
      refetch();
      toast({
        title: 'Employee Updated',
        description: `${employee.name} has been updated successfully.`,
      });
    } else {
      await addEmployee(employee as Employee);
      refetch();
      toast({
        title: 'Employee Added',
        description: `${employee.name} has been added successfully.`,
      });
    }
    setIsFormOpen(false);
  };
  
  const handleFormCancel = () => {
    setIsFormOpen(false);
  };
  
  const handleDeleteEmployee = async (id: string) => {
    if (!canManageEmployees) {
      toast({
        title: 'Permission Denied',
        description: 'You need super admin privileges to delete employees.',
        variant: 'destructive',
      });
      return;
    }
    
    await deleteEmployee(id);
    refetch();
    toast({
      title: 'Employee Deleted',
      description: 'Employee has been removed from the list.',
    });
  };
  
  const handleToggleVoting = () => {
    if (!canManageVoting) {
      toast({
        title: 'Permission Denied',
        description: 'You need admin privileges to manage voting status.',
        variant: 'destructive',
      });
      return;
    }
    
    if (votingData.isOpen) {
      closeVoting();
      toast({
        title: 'Voting Closed',
        description: 'Voting has been manually closed.',
      });
    } else {
      openVoting();
      toast({
        title: 'Voting Opened',
        description: 'Voting is now active.',
      });
    }
  };
  
  const handleDownloadResults = () => {
    const csvContent = [
      ['Employee Name', 'Title', 'Votes'],
      ...employees.map(emp => [emp.name, emp.title, emp.votes.toString()]),
    ]
      .map(row => row.join(','))
      .join('\n');
    
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'employee-voting-results.csv';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    toast({
      title: 'Download Complete',
      description: 'Voting results have been downloaded as CSV.',
    });
  };

  const handleCsvUpload = (employees: Partial<Employee>[]) => {
    if (!canManageEmployees) {
      toast({
        title: 'Permission Denied',
        description: 'You need super admin privileges to add employees.',
        variant: 'destructive',
      });
      return;
    }
    
    console.log('Received employees from CSV:', employees);
    
    if (!employees || employees.length === 0) {
      toast({
        title: 'Error',
        description: 'No valid employee data found in CSV file.',
        variant: 'destructive',
      });
      return;
    }
    
    let addedCount = 0;

    employees.forEach(emp => {
      // Ensure we have the minimum required fields
      if (emp.name && emp.title) {
        const newEmployee: Employee = {
          id: emp.id || Math.random().toString(36).substring(2, 15),
          name: emp.name,
          title: emp.title,
          description: emp.description || `${emp.name} is a ${emp.title}`,
          image_url: emp.image_url || `https://api.dicebear.com/7.x/initials/svg?seed=${encodeURIComponent(emp.name)}`,
          votes: 0
        };
        
        addEmployee(newEmployee);
        addedCount++;
      }
    });
    
    toast({
      title: 'Employees Added',
      description: `Successfully added ${addedCount} employees to the system.`,
      variant: addedCount > 0 ? 'default' : 'destructive',
    });
  };

  if (!currentUser?.isAdmin) {
    return null;
  }

  return (
    <div>
      <AdminHeader 
        currentUser={currentUser} 
        onOpenRoleForm={() => setIsRoleFormOpen(true)} 
      />
      
      <Tabs value={activeTab} onValueChange={setActiveTab} className="mb-8">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="employees">Manage Employees</TabsTrigger>
          <TabsTrigger value="voting">Voting Controls</TabsTrigger>
          <TabsTrigger value="analytics">Data Analytics</TabsTrigger>
        </TabsList>
        
        <TabsContent value="employees" className="space-y-4">
          <EmployeeManagement 
            employees={employees}
            canManageEmployees={canManageEmployees}
            onAddEmployee={handleAddEmployee}
            onEditEmployee={handleEditEmployee}
            onDeleteEmployee={handleDeleteEmployee}
            refetch={refetch}
          />
        </TabsContent>
        
        <TabsContent value="voting" className="space-y-6">
          <VotingControls 
            votingData={votingData}
            canManageVoting={canManageVoting}
            employees={employees}
            onToggleVoting={handleToggleVoting}
            onDownloadResults={handleDownloadResults}
          />
        </TabsContent>

        <TabsContent value="analytics" className="space-y-4">
          <AdminAnalytics employees={employees} />
        </TabsContent>
      </Tabs>
      
      <Dialog open={isFormOpen} onOpenChange={setIsFormOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>{currentEmployee ? 'Edit Employee' : 'Add New Employee'}</DialogTitle>
          </DialogHeader>
          <AdminEmployeeForm 
            employee={currentEmployee} 
            onSubmit={handleFormSubmit} 
            onCancel={handleFormCancel} 
          />
        </DialogContent>
      </Dialog>

      <Dialog open={isRoleFormOpen} onOpenChange={setIsRoleFormOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Manage Admin Roles</DialogTitle>
          </DialogHeader>
          <AdminRoleForm onClose={() => setIsRoleFormOpen(false)} />
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default AdminPage;
