
import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from 'react-router-dom';
import { Play, Pause, BarChart3, Download } from 'lucide-react';
import { VotingData, Employee } from '@/types';

interface VotingControlsProps {
  votingData: VotingData;
  canManageVoting: boolean;
  employees: Employee[];
  onToggleVoting: () => void;
  onDownloadResults: () => void;
}

const VotingControls: React.FC<VotingControlsProps> = ({
  votingData,
  canManageVoting,
  employees,
  onToggleVoting,
  onDownloadResults,
}) => {
  const navigate = useNavigate();

  return (
    <>
      <Card>
        <CardHeader>
          <CardTitle>Voting Status</CardTitle>
          <CardDescription>Control the status of the voting period</CardDescription>
        </CardHeader>
        <CardContent className="flex justify-between items-center">
          <div>
            <p className="text-sm font-medium">Current Status:</p>
            <p className={`text-lg font-bold ${votingData.isOpen ? 'text-green-600' : 'text-red-600'}`}>
              {votingData.isOpen ? 'Voting Open' : 'Voting Closed'}
            </p>
          </div>
          {canManageVoting ? (
            <Button onClick={onToggleVoting} className={votingData.isOpen ? 'bg-red-600 hover:bg-red-700' : 'bg-green-600 hover:bg-green-700'}>
              {votingData.isOpen ? (
                <>
                  <Pause className="mr-2 h-4 w-4" /> Close Voting
                </>
              ) : (
                <>
                  <Play className="mr-2 h-4 w-4" /> Open Voting
                </>
              )}
            </Button>
          ) : (
            <Button disabled className="opacity-50">
              {votingData.isOpen ? (
                <>
                  <Pause className="mr-2 h-4 w-4" /> Close Voting
                </>
              ) : (
                <>
                  <Play className="mr-2 h-4 w-4" /> Open Voting
                </>
              )}
            </Button>
          )}
        </CardContent>
      </Card>
      
      <Card>
        <CardHeader>
          <CardTitle>Results & Analytics</CardTitle>
          <CardDescription>View and export voting results</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="flex flex-wrap gap-4">
            <Button variant="outline" onClick={() => navigate('/results')}>
              <BarChart3 className="mr-2 h-4 w-4" /> View Results
            </Button>
            <Button variant="outline" onClick={onDownloadResults}>
              <Download className="mr-2 h-4 w-4" /> Export as CSV
            </Button>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-4">
            <Card>
              <CardHeader className="pb-2">
                <CardTitle className="text-lg">Total Votes</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-3xl font-bold">
                  {employees.reduce((acc, emp) => acc + emp.votes, 0)}
                </p>
              </CardContent>
            </Card>
            <Card>
              <CardHeader className="pb-2">
                <CardTitle className="text-lg">Nominees</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-3xl font-bold">
                  {employees.length}
                </p>
              </CardContent>
            </Card>
            <Card>
              <CardHeader className="pb-2">
                <CardTitle className="text-lg">Current Leader</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-xl font-bold truncate">
                  {employees.length > 0 
                    ? [...employees].sort((a, b) => b.votes - a.votes)[0].name
                    : "No votes yet"}
                </p>
              </CardContent>
            </Card>
          </div>
        </CardContent>
      </Card>
    </>
  );
};

export default VotingControls;
