
import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '@/components/ui/alert-dialog';
import { PlusCircle, Edit, Trash, Trophy } from 'lucide-react';
import { Employee } from '@/types';
import CsvUpload from '@/components/CsvUpload';

interface EmployeeManagementProps {
  employees: Employee[];
  canManageEmployees: boolean;
  onAddEmployee: () => void;
  onEditEmployee: (employee: Employee) => void;
  onDeleteEmployee: (id: string) => void;
  refetch: () => void;
}

const EmployeeManagement: React.FC<EmployeeManagementProps> = ({
  employees,
  canManageEmployees,
  onAddEmployee,
  onEditEmployee,
  onDeleteEmployee,
  refetch,
}) => {
  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between pb-2">
        <div>
          <CardTitle>Employees</CardTitle>
          <CardDescription>Manage employees eligible for voting</CardDescription>
        </div>
        {canManageEmployees && (
          <div className="flex items-center gap-4">
            <CsvUpload onUpload={() => refetch()} />
            <Button onClick={onAddEmployee}>
              <PlusCircle className="mr-2 h-4 w-4" /> Add Employee
            </Button>
          </div>
        )}
      </CardHeader>
      <CardContent>
        {employees.length > 0 ? (
          <div className="divide-y">
            {employees.map((employee) => (
              <div key={employee.id} className="py-4 flex items-center justify-between">
                <div className="flex items-center">
                  <div className="w-10 h-10 rounded-full overflow-hidden bg-gray-100 mr-3">
                    <img 
                      src={employee.image_url} 
                      alt={employee.name} 
                      className="w-full h-full object-cover"
                      onError={(e) => {
                        (e.target as HTMLImageElement).src = `https://api.dicebear.com/7.x/initials/svg?seed=${encodeURIComponent(employee.name)}`;
                      }}
                    />
                  </div>
                  <div>
                    <p className="font-medium">{employee.name}</p>
                    <p className="text-sm text-muted-foreground">{employee.title}</p>
                  </div>
                </div>
                <div className="flex items-center gap-2">
                  <p className="mr-4 text-sm"><Trophy className="inline h-4 w-4 mr-1 text-yellow-500" /> {employee.votes} votes</p>
                  {canManageEmployees && (
                    <>
                      <Button variant="outline" size="icon" onClick={() => onEditEmployee(employee)}>
                        <Edit className="h-4 w-4" />
                        <span className="sr-only">Edit</span>
                      </Button>
                      <AlertDialog>
                        <AlertDialogTrigger asChild>
                          <Button variant="outline" size="icon" className="text-destructive hover:text-destructive">
                            <Trash className="h-4 w-4" />
                            <span className="sr-only">Delete</span>
                          </Button>
                        </AlertDialogTrigger>
                        <AlertDialogContent>
                          <AlertDialogHeader>
                            <AlertDialogTitle>Delete Employee</AlertDialogTitle>
                            <AlertDialogDescription>
                              Are you sure you want to remove {employee.name} from the voting list? This action cannot be undone.
                            </AlertDialogDescription>
                          </AlertDialogHeader>
                          <AlertDialogFooter>
                            <AlertDialogCancel>Cancel</AlertDialogCancel>
                            <AlertDialogAction onClick={() => onDeleteEmployee(employee.id)}>
                              Delete
                            </AlertDialogAction>
                          </AlertDialogFooter>
                        </AlertDialogContent>
                      </AlertDialog>
                    </>
                  )}
                </div>
              </div>
            ))}
          </div>
        ) : (
          <p className="py-4 text-center text-muted-foreground">No employees have been added yet.</p>
        )}
      </CardContent>
    </Card>
  );
};

export default EmployeeManagement;
