
import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardFooter } from '@/components/ui/card';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Trophy, ThumbsUp } from 'lucide-react';
import { useVotingStore } from '../store/votingStore';
import { Employee } from '../types';
import { useToast } from '@/components/ui/use-toast';
import { supabase } from '@/integrations/supabase/client';

interface EmployeeCardProps {
  employee: Employee;
  showVotes?: boolean;
  onVoteSuccess?: () => void;
}

const EmployeeCard: React.FC<EmployeeCardProps> = ({ 
  employee, 
  showVotes = false, 
  onVoteSuccess 
}) => {
  const { castVote, currentUser, votingData } = useVotingStore();
  const { toast } = useToast();
  const [isVoting, setIsVoting] = useState(false);

  const saveVoteAnalytics = async (email: string) => {
    // Get user agent information
    const userAgent = navigator.userAgent;
    let browser = "Unknown";
    let device = "Unknown";
    let os = "Unknown";

    // Basic browser detection
    if (userAgent.indexOf("Firefox") > -1) {
      browser = "Firefox";
    } else if (userAgent.indexOf("SamsungBrowser") > -1) {
      browser = "Samsung Browser";
    } else if (userAgent.indexOf("Opera") > -1 || userAgent.indexOf("OPR") > -1) {
      browser = "Opera";
    } else if (userAgent.indexOf("Trident") > -1) {
      browser = "Internet Explorer";
    } else if (userAgent.indexOf("Edge") > -1 || userAgent.indexOf("Edg") > -1) {
      browser = "Edge";
    } else if (userAgent.indexOf("Chrome") > -1) {
      browser = "Chrome";
    } else if (userAgent.indexOf("Safari") > -1) {
      browser = "Safari";
    }

    // Basic device detection
    if (/(tablet|ipad|playbook|silk)|(android(?!.*mobi))/i.test(userAgent)) {
      device = "Tablet";
    } else if (/Mobile|Android|iP(hone|od)|IEMobile|BlackBerry|Kindle|Silk-Accelerated|(hpw|web)OS|Opera M(obi|ini)/.test(userAgent)) {
      device = "Mobile";
    } else {
      device = "Desktop";
    }

    // Basic OS detection
    if (userAgent.indexOf("Win") > -1) {
      os = "Windows";
    } else if (userAgent.indexOf("Mac") > -1) {
      os = "MacOS";
    } else if (userAgent.indexOf("Android") > -1) {
      os = "Android";
    } else if (userAgent.indexOf("Linux") > -1) {
      os = "Linux";
    } else if (userAgent.indexOf("iPhone") > -1 || userAgent.indexOf("iPad") > -1) {
      os = "iOS";
    }

    // Save vote analytics to Supabase
    await supabase.from('vote_analytics').insert([
      {
        user_email: email,
        user_agent: userAgent,
        browser,
        device,
        os
      }
    ]);
  };

  const sendEmailConfirmation = async (email: string, employeeName: string) => {
    try {
      // Update the flag in the user_votes table
      await supabase
        .from('user_votes')
        .update({ is_sent_confirmation: true })
        .eq('user_email', email);
      
      // In a real app, you would call an edge function to send the email
      console.log(`Email confirmation would be sent to ${email} for voting for ${employeeName}`);
      
      // For demo purposes, we'll just update the flag and show a toast
      toast({
        title: "Vote Confirmation Email",
        description: `A confirmation email would be sent to ${email}`,
      });
    } catch (error) {
      console.error("Error updating confirmation status:", error);
    }
  };

  const handleVote = async () => {
    if (!currentUser) {
      toast({
        title: "Authentication Required",
        description: "Please sign in to cast your vote",
        variant: "destructive",
      });
      return;
    }

    if (currentUser.hasVoted) {
      toast({
        title: "Already Voted",
        description: "You have already cast your vote",
        variant: "destructive",
      });
      return;
    }

    if (!votingData.isOpen) {
      toast({
        title: "Voting Closed",
        description: "The voting period has ended",
        variant: "destructive",
      });
      return;
    }

    setIsVoting(true);
    
    try {
      await castVote(employee.id);
      
      // Record vote analytics
      if (currentUser.email) {
        await saveVoteAnalytics(currentUser.email);
      }
      
      toast({
        title: "Vote Registered!",
        description: `You voted for ${employee.name}`,
        variant: "default",
      });
      
      // Send email confirmation
      if (currentUser.email) {
        await sendEmailConfirmation(currentUser.email, employee.name);
      }
      
      if (onVoteSuccess) {
        onVoteSuccess();
      }
    } catch (error) {
      console.error("Voting error:", error);
      toast({
        title: "Voting Failed",
        description: error instanceof Error ? error.message : "An error occurred while processing your vote",
        variant: "destructive",
      });
    } finally {
      setIsVoting(false);
    }
  };

  const isVotedByUser = currentUser?.votedFor === employee.id;
  const initials = employee.name
    .split(' ')
    .map(word => word[0])
    .join('');

  return (
    <Card className={`overflow-hidden transition-all duration-300 hover:shadow-lg ${isVotedByUser ? 'ring-2 ring-primary' : ''}`}>
      <CardHeader className="p-4 pb-0">
        <div className="flex items-center justify-center">
          <Avatar className="h-24 w-24">
            <AvatarImage src={employee.image_url} alt={employee.name} />
            <AvatarFallback className="text-xl">{initials}</AvatarFallback>
          </Avatar>
        </div>
      </CardHeader>
      <CardContent className="p-4 text-center">
        <h3 className="font-bold text-lg mb-1">{employee.name}</h3>
        <p className="text-sm text-muted-foreground mb-2">{employee.title}</p>
        <p className="text-sm italic">"{employee.description}"</p>
      </CardContent>
      <CardFooter className="p-4 pt-0 flex flex-col gap-2">
        {showVotes && (
          <div className="flex items-center justify-center gap-1 text-sm font-medium mb-2">
            <Trophy className="h-4 w-4 text-yellow-500" />
            <span>{employee.votes} votes</span>
          </div>
        )}
        
        <Button 
          onClick={handleVote}
          disabled={!currentUser || currentUser.hasVoted || !votingData.isOpen || isVoting}
          className={`w-full ${isVotedByUser ? 'bg-green-600 hover:bg-green-700' : ''} ${isVoting ? 'animate-pulse-vote' : ''}`}
        >
          {isVotedByUser ? (
            <>
              <ThumbsUp className="mr-2 h-4 w-4" /> Voted
            </>
          ) : (
            "Cast Vote"
          )}
        </Button>
      </CardFooter>
    </Card>
  );
};

export default EmployeeCard;
