
import React, { useRef } from 'react';
import { Button } from '@/components/ui/button';
import { Upload, Download } from 'lucide-react';
import { useToast } from '@/components/ui/use-toast';
import { getEmployeeCsvTemplate } from '@/utils/csvTemplate';
import { Employee } from '@/types';

interface CsvUploadProps {
  onUpload: (employees: Partial<Employee>[]) => void;
}

const CsvUpload: React.FC<CsvUploadProps> = ({ onUpload }) => {
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (e) => {
      try {
        const text = e.target?.result as string;
        const rows = text.split('\n');
        const headers = rows[0].split(',');
        
        const employees = rows.slice(1)
          .filter(row => row.trim()) // Skip empty rows
          .map(row => {
            const values = row.split(',');
            const employee: Partial<Employee> = {
              id: Math.random().toString(36).substring(2, 15),
              votes: 0
            };
            
            headers.forEach((header, index) => {
              const trimmedHeader = header.trim().toLowerCase();
              const value = values[index]?.trim();
              
              if (value) {
                // Map CSV headers to employee properties
                if (trimmedHeader === 'name') {
                  employee.name = value;
                } else if (trimmedHeader === 'title') {
                  employee.title = value;
                } else if (trimmedHeader === 'description') {
                  employee.description = value;
                } else if (trimmedHeader === 'imageurl') {
                  employee.image_url = value;
                }
              }
            });
            
            // Ensure required fields are present
            if (!employee.name || !employee.title) {
              console.warn('Skipping row with missing required fields:', row);
              return null;
            }
            
            // Set default description if missing
            if (!employee.description) {
              employee.description = 'No description provided';
            }
            
            // Set default image URL if missing
            if (!employee.image_url) {
              employee.image_url = `https://api.dicebear.com/7.x/initials/svg?seed=${encodeURIComponent(employee.name || '')}`;
            }
            
            return employee;
          })
          .filter(Boolean) as Partial<Employee>[];

        console.log('Parsed employees:', employees);
        if (employees.length === 0) {
          toast({
            title: 'No valid employees found',
            description: 'Please check your CSV format and try again.',
            variant: 'destructive',
          });
          return;
        }

        onUpload(employees);
        toast({
          title: 'Success',
          description: `${employees.length} employees imported successfully.`,
        });
        
        // Reset file input after successful upload
        if (fileInputRef.current) {
          fileInputRef.current.value = '';
        }
      } catch (error) {
        console.error('CSV parsing error:', error);
        toast({
          title: 'Error',
          description: 'Failed to parse CSV file. Please check the format.',
          variant: 'destructive',
        });
      }
    };
    reader.readAsText(file);
  };

  const downloadTemplate = () => {
    const template = getEmployeeCsvTemplate();
    const blob = new Blob([template], { type: 'text/csv' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'employee-template.csv';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  return (
    <div className="space-y-4">
      <div className="flex items-center gap-4">
        <Button
          variant="outline"
          onClick={() => fileInputRef.current?.click()}
          className="flex items-center gap-2"
        >
          <Upload className="h-4 w-4" />
          Import CSV
        </Button>
        <Button
          variant="outline"
          onClick={downloadTemplate}
          className="flex items-center gap-2"
        >
          <Download className="h-4 w-4" />
          Download Template
        </Button>
      </div>
      <input
        type="file"
        ref={fileInputRef}
        onChange={handleFileChange}
        accept=".csv"
        className="hidden"
      />
    </div>
  );
};

export default CsvUpload;
