
import React, { useState } from 'react';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { useToast } from '@/components/ui/use-toast';
import { supabase } from '@/integrations/supabase/client';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { AdminRole } from '@/types';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Trash } from 'lucide-react';

const formSchema = z.object({
  email: z.string().email('Must be a valid email'),
  role: z.enum(['super_admin', 'voting_admin', 'viewer']),
});

type FormValues = z.infer<typeof formSchema>;

interface Admin {
  id: string;
  email: string;
  role: AdminRole;
  created_at: string;
}

interface AdminRoleFormProps {
  onClose: () => void;
}

const AdminRoleForm: React.FC<AdminRoleFormProps> = ({ onClose }) => {
  const { toast } = useToast();
  const [admins, setAdmins] = useState<Admin[]>([]);
  const [loading, setLoading] = useState(true);

  const form = useForm<FormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      email: '',
      role: 'viewer',
    },
  });

  // Fetch current admins when component mounts
  React.useEffect(() => {
    const fetchAdmins = async () => {
      try {
        const { data, error } = await supabase
          .from('admins')
          .select('*')
          .order('created_at', { ascending: false });
          
        if (error) throw error;
        setAdmins(data as Admin[]);
      } catch (error) {
        console.error('Error fetching admins:', error);
        toast({
          title: 'Error',
          description: 'Failed to fetch admin users',
          variant: 'destructive',
        });
      } finally {
        setLoading(false);
      }
    };
    
    fetchAdmins();
  }, [toast]);

  const onSubmit = async (data: FormValues) => {
    try {
      // Check if admin already exists
      const existingAdmin = admins.find(admin => admin.email.toLowerCase() === data.email.toLowerCase());
      
      if (existingAdmin) {
        // Update existing admin role
        const { error } = await supabase
          .from('admins')
          .update({ role: data.role })
          .eq('id', existingAdmin.id);
          
        if (error) throw error;
        
        // Update local state
        setAdmins(admins.map(admin => 
          admin.id === existingAdmin.id ? { ...admin, role: data.role } : admin
        ));
        
        toast({
          title: 'Admin Updated',
          description: `${data.email}'s role updated to ${data.role}`,
        });
      } else {
        // Add new admin
        const { data: newAdmin, error } = await supabase
          .from('admins')
          .insert([{ email: data.email, role: data.role }])
          .select()
          .single();
          
        if (error) throw error;
        
        // Update local state
        setAdmins([newAdmin as Admin, ...admins]);
        
        toast({
          title: 'Admin Added',
          description: `${data.email} added as ${data.role}`,
        });
      }
      
      // Reset form
      form.reset({
        email: '',
        role: 'viewer',
      });
    } catch (error: any) {
      console.error('Error managing admin:', error);
      toast({
        title: 'Error',
        description: error.message || 'Failed to update admin role',
        variant: 'destructive',
      });
    }
  };

  const handleDeleteAdmin = async (id: string, email: string) => {
    try {
      const { error } = await supabase
        .from('admins')
        .delete()
        .eq('id', id);
        
      if (error) throw error;
      
      // Update local state
      setAdmins(admins.filter(admin => admin.id !== id));
      
      toast({
        title: 'Admin Removed',
        description: `${email} has been removed from admin list`,
      });
    } catch (error: any) {
      console.error('Error deleting admin:', error);
      toast({
        title: 'Error',
        description: error.message || 'Failed to remove admin',
        variant: 'destructive',
      });
    }
  };

  return (
    <div className="space-y-6">
      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
          <FormField
            control={form.control}
            name="email"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Email</FormLabel>
                <FormControl>
                  <Input placeholder="admin@company.com" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="role"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Role</FormLabel>
                <FormControl>
                  <Select 
                    value={field.value} 
                    onValueChange={(value: AdminRole) => field.onChange(value)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select a role" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="super_admin">Super Admin (Full Powers)</SelectItem>
                      <SelectItem value="voting_admin">Voting Admin (Manage Votes Only)</SelectItem>
                      <SelectItem value="viewer">Viewer (Read Only)</SelectItem>
                    </SelectContent>
                  </Select>
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <Button type="submit" className="w-full">
            Add/Update Admin
          </Button>
        </form>
      </Form>
      
      <div>
        <h3 className="text-lg font-medium mb-2">Current Admins</h3>
        {loading ? (
          <p className="text-center py-4 text-muted-foreground">Loading admins...</p>
        ) : (
          <>
            {admins.length > 0 ? (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Email</TableHead>
                    <TableHead>Role</TableHead>
                    <TableHead className="w-[100px]">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {admins.map((admin) => (
                    <TableRow key={admin.id}>
                      <TableCell>{admin.email}</TableCell>
                      <TableCell>
                        {admin.role === 'super_admin' ? 'Super Admin' : 
                         admin.role === 'voting_admin' ? 'Voting Admin' : 'Viewer'}
                      </TableCell>
                      <TableCell>
                        <Button 
                          variant="ghost" 
                          size="sm"
                          onClick={() => handleDeleteAdmin(admin.id, admin.email)}
                        >
                          <Trash className="h-4 w-4 text-destructive" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            ) : (
              <p className="text-center py-4 text-muted-foreground">No admins found.</p>
            )}
          </>
        )}
      </div>
      
      <div className="flex justify-end">
        <Button variant="outline" onClick={onClose}>Close</Button>
      </div>
    </div>
  );
};

export default AdminRoleForm;
