
import React, { useState } from 'react';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Employee } from '../types';

const formSchema = z.object({
  name: z.string().min(2, 'Name is required'),
  title: z.string().min(2, 'Job title is required'),
  description: z.string().min(5, 'Description is required'),
  image_url: z.string().url('Must be a valid URL').or(z.string().length(0)),
});

type FormValues = z.infer<typeof formSchema>;

interface Props {
  employee?: Employee;
  onSubmit: (data: Partial<Employee>) => void;
  onCancel: () => void;
}

const AdminEmployeeForm: React.FC<Props> = ({ employee, onSubmit, onCancel }) => {
  const isEditing = !!employee;

  const form = useForm<FormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      name: employee?.name || '',
      title: employee?.title || '',
      description: employee?.description || '',
      image_url: employee?.image_url || '',
    },
  });

  const handleSubmit = (data: FormValues) => {
    // If image_url is empty, use a placeholder
    if (!data.image_url) {
      data.image_url = `https://api.dicebear.com/7.x/initials/svg?seed=${encodeURIComponent(data.name)}`;
    }

    onSubmit({
      id: employee?.id || Math.random().toString(36).substring(2, 15),
      ...data,
      votes: employee?.votes || 0,
    });
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="name"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Full Name</FormLabel>
              <FormControl>
                <Input placeholder="John Doe" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="title"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Job Title</FormLabel>
              <FormControl>
                <Input placeholder="Senior Developer" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="description"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Short Description</FormLabel>
              <FormControl>
                <Textarea placeholder="Always delivering excellence!" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="image_url"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Image URL (optional)</FormLabel>
              <FormControl>
                <Input placeholder="https://example.com/image.jpg" {...field} />
              </FormControl>
              <FormMessage />
              <p className="text-xs text-muted-foreground">Leave empty to use generated avatar</p>
            </FormItem>
          )}
        />

        <div className="flex justify-end gap-2 pt-2">
          <Button variant="outline" type="button" onClick={onCancel}>
            Cancel
          </Button>
          <Button type="submit">
            {isEditing ? 'Update' : 'Add'} Employee
          </Button>
        </div>
      </form>
    </Form>
  );
};

export default AdminEmployeeForm;
