
import React from 'react';
import { useQuery } from '@tanstack/react-query';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { PieChart, Pie, Cell, ResponsiveContainer, BarChart, Bar, XAxis, YAxis, Tooltip, Legend } from 'recharts';
import { supabase } from '@/integrations/supabase/client';
import { Employee, VoteAnalytics } from '../types';

interface AnalyticsProps {
  employees: Employee[];
}

const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884d8', '#82ca9d'];

const AdminAnalytics: React.FC<AnalyticsProps> = ({ employees }) => {
  const { data: analytics, isLoading } = useQuery({
    queryKey: ['vote-analytics'],
    queryFn: async () => {
      const { data, error } = await supabase
        .from('vote_analytics')
        .select('*')
        .order('vote_timestamp', { ascending: false });

      if (error) throw error;
      return data as VoteAnalytics[];
    },
  });

  // Calculate votes percentage for pie chart
  const pieData = employees.map(emp => ({
    name: emp.name,
    value: emp.votes
  })).filter(item => item.value > 0);

  // Calculate browser statistics
  const browserStats = analytics?.reduce((acc, item) => {
    const browser = item.browser || 'Unknown';
    acc[browser] = (acc[browser] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);

  const browserData = browserStats ? Object.entries(browserStats).map(([name, value]) => ({ name, value })) : [];

  // Calculate device statistics
  const deviceStats = analytics?.reduce((acc, item) => {
    const device = item.device || 'Unknown';
    acc[device] = (acc[device] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);

  const deviceData = deviceStats ? Object.entries(deviceStats).map(([name, value]) => ({ name, value })) : [];

  // Calculate votes by hour of day for a time series chart
  const votesByHour = analytics?.reduce((acc, item) => {
    const hour = new Date(item.vote_timestamp).getHours();
    acc[hour] = (acc[hour] || 0) + 1;
    return acc;
  }, {} as Record<number, number>);

  const hourlyData = votesByHour ? Array.from({ length: 24 }, (_, i) => ({
    hour: i,
    votes: votesByHour[i] || 0,
    label: `${i}:00`
  })) : [];

  if (isLoading) {
    return <div className="flex justify-center items-center h-64">Loading analytics data...</div>;
  }

  return (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Vote Distribution</CardTitle>
            <CardDescription>Percentage of votes per nominee</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-80">
              {pieData.length > 0 ? (
                <ResponsiveContainer width="100%" height="100%">
                  <PieChart>
                    <Pie
                      data={pieData}
                      cx="50%"
                      cy="50%"
                      labelLine={false}
                      label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                      outerRadius={80}
                      fill="#8884d8"
                      dataKey="value"
                    >
                      {pieData.map((entry, index) => (
                        <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                      ))}
                    </Pie>
                    <Tooltip />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              ) : (
                <div className="flex justify-center items-center h-full text-center text-muted-foreground">
                  No voting data available yet
                </div>
              )}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Peak Voting Times</CardTitle>
            <CardDescription>Number of votes by hour of day</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-80">
              {hourlyData.length > 0 && analytics?.length ? (
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart data={hourlyData}>
                    <XAxis dataKey="label" />
                    <YAxis />
                    <Tooltip />
                    <Legend />
                    <Bar dataKey="votes" fill="#8884d8" name="Votes" />
                  </BarChart>
                </ResponsiveContainer>
              ) : (
                <div className="flex justify-center items-center h-full text-center text-muted-foreground">
                  No time data available yet
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Browser Statistics</CardTitle>
            <CardDescription>Votes by browser type</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-80">
              {browserData.length > 0 ? (
                <ResponsiveContainer width="100%" height="100%">
                  <PieChart>
                    <Pie
                      data={browserData}
                      cx="50%"
                      cy="50%"
                      labelLine={false}
                      label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                      outerRadius={80}
                      fill="#8884d8"
                      dataKey="value"
                    >
                      {browserData.map((entry, index) => (
                        <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                      ))}
                    </Pie>
                    <Tooltip />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              ) : (
                <div className="flex justify-center items-center h-full text-center text-muted-foreground">
                  No browser data available yet
                </div>
              )}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Device Statistics</CardTitle>
            <CardDescription>Votes by device type</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-80">
              {deviceData.length > 0 ? (
                <ResponsiveContainer width="100%" height="100%">
                  <PieChart>
                    <Pie
                      data={deviceData}
                      cx="50%"
                      cy="50%"
                      labelLine={false}
                      label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                      outerRadius={80}
                      fill="#8884d8"
                      dataKey="value"
                    >
                      {deviceData.map((entry, index) => (
                        <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                      ))}
                    </Pie>
                    <Tooltip />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              ) : (
                <div className="flex justify-center items-center h-full text-center text-muted-foreground">
                  No device data available yet
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Recent Votes</CardTitle>
          <CardDescription>Last 20 votes cast</CardDescription>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Date & Time</TableHead>
                <TableHead>User</TableHead>
                <TableHead>Browser</TableHead>
                <TableHead>Device</TableHead>
                <TableHead>OS</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {analytics && analytics.length > 0 ? (
                analytics.slice(0, 20).map((vote) => (
                  <TableRow key={vote.id}>
                    <TableCell>
                      {new Date(vote.vote_timestamp).toLocaleString()}
                    </TableCell>
                    <TableCell>{vote.user_email}</TableCell>
                    <TableCell>{vote.browser || 'Unknown'}</TableCell>
                    <TableCell>{vote.device || 'Unknown'}</TableCell>
                    <TableCell>{vote.os || 'Unknown'}</TableCell>
                  </TableRow>
                ))
              ) : (
                <TableRow>
                  <TableCell colSpan={5} className="text-center py-4">
                    No vote data available yet
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>
    </div>
  );
};

export default AdminAnalytics;
